/*
 * This file is part of Renewed City Growth, a GameScript for OpenTTD.
 * Original code written by kormer, Zuu, TrueBrain and Aphid.
 * Copyright (C) 2013-2014 keoz
 *
 * Renewed City Growth is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, version 2 of the
 * License.
 *
 * Renewed City Growth is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Renewed City Growth; If not, see
 * <http://www.gnu.org/licenses/> or write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 *
 */


require("version.nut");
require("cargo.nut");
require("town.nut");
require("strings.nut");
require("industry_builder.nut")

// Import SuperLib for GameScript
import("util.superlib", "SuperLib", 38);
Result <- SuperLib.Result;
Log <- SuperLib.Log;
Helper <- SuperLib.Helper;
Tile <- SuperLib.Tile;
Direction <- SuperLib.Direction;
Town <- SuperLib.Town;
Industry <- SuperLib.Industry;
Story <- SuperLib.Story;

class MainClass extends GSController 
{
	_towns = null;
	_current_date = null;
	_current_month = null;
	_current_year = null;
	_gs_init_done = null;
	load_saved_data = null;
	current_save_version = null;
	industryBuilder = null;

	constructor() {
		this._towns = null;
		this._current_date = 0;
		this._current_month = 0;
		this._current_year = 0;
		this._gs_init_done = false;
		::_load_saved_data <- true;      // Set this to 'false' to reset all data when loading
		::_current_save_version <- 1;    // Ensure compatibility between revisions
		::_loaded_save_version <- null;
		::_town_data_table <- {};
		
		this.industryBuilder = IndustryBuilder();
		this.industryBuilder.setPrefix(">>>");
		this.industryBuilder.setSeparator(":"); 
		switch (GSController.GetSetting("log_level"))
		{
		case 1: 
			this.industryBuilder.setRemoveSigns(true);  
			this.industryBuilder.setPlaceSignAfterBuild(1); //quiet
			break;
		case 2: 
			this.industryBuilder.setRemoveSigns(true);  
			this.industryBuilder.setPlaceSignAfterBuild(2); //verbose
			break;
		case 3: 
			this.industryBuilder.setRemoveSigns(false); 
			this.industryBuilder.setPlaceSignAfterBuild(3); //debug all
			this.industryBuilder.dumpIndustryNumbers();
			break;
		}
		this.industryBuilder.setTryCount( 12 );		
		this.industryBuilder.setYearCount( 20 );		//todo: as parameter


	}
}

function MainClass::Start()
{
	// Initializing the script
	local start_tick = GSController.GetTick();
	GSGame.Pause();
	Log.Info("Script initialisation...", Log.LVL_INFO);
	this.Init();
	GSGame.Unpause();

	local setup_duration = GSController.GetTick() - start_tick;
	Log.Info("Game setup done.", Log.LVL_INFO);
	Log.Info("Setup took " + setup_duration + " ticks.", Log.LVL_DEBUG);
	Log.Info("Happy playing !", Log.LVL_INFO);

	// Wait for the game to start
	GSController.Sleep(1);

	local last_loop_date = GSDate.GetCurrentDate();

	// Main loop
	while (true) {

//TLIB
		local loop_start_tick = GSController.GetTick();

		// Handle incoming messages from OpenTTD
		this.HandleEvents();

		// Reached new year/month?
		local current_date = GSDate.GetCurrentDate();
		if (last_loop_date != null) {
			local year = GSDate.GetYear(current_date);
			local month = GSDate.GetMonth(current_date);
			if (year != GSDate.GetYear(last_loop_date)) {
				this.EndOfYear(year);
			}
			if (month != GSDate.GetMonth(last_loop_date)) {
				this.EndOfMonth(year);
			}
		}
		last_loop_date = current_date;
	
		// Loop with a frequency of five days
		local ticks_used = GSController.GetTick() - loop_start_tick;
		GSController.Sleep(max(1, 5 * 74 - ticks_used));
//TLIB

		this.HandleEvents();
		this.ManageTowns();
		this.Sleep(60);
	}
}

function MainClass::Init()
{
	// Check game settings
	GSGameSettings.SetValue("economy.town_growth_rate", 2);
	GSGameSettings.SetValue("economy.fund_buildings", 0);

	// Set current date
	this._current_date = GSDate.GetCurrentDate();
	this._current_month = GSDate.GetMonth(this._current_date);
	this._current_year = GSDate.GetYear(this._current_date);

//TLIB
	industryBuilder.init();

	industryBuilder.readFromMap();
	this.industryBuilder.dump();
	
//TLIB


	// Initialize cargo lists and variables
	InitCargoLists();

	/* Check whether saved data are in the current save
	 * format.
	 */
	if (::_loaded_save_version != ::_current_save_version) {
		::_load_saved_data = false;
	}

	if (!::_load_saved_data) {
		Helper.ClearAllSigns();
	}

	// Create the towns list
	Log.Info("Create town list ... (can take a while on large maps)", Log.LVL_INFO);
	this._towns = this.CreateTownList();

	this._gs_init_done = true;
	// Freeing the town data table but not as null (necessary for town foundation)
	::_town_data_table = {};
}

function MainClass::HandleEvents()
{
	while (GSEventController.IsEventWaiting()) {
		local event = GSEventController.GetNextEvent();

		switch (event.GetEventType()) {
		// On town founding, add a new GoalTown instance
		case GSEvent.ET_TOWN_FOUNDED:
			event = GSEventTownFounded.Convert(event);
			local town_id = event.GetTownID();
			if (GSTown.IsValidTown(town_id)) this.UpdateTownList(town_id);
			break;

		default: break;
		}
	}
}

function MainClass::Save()
{
	Log.Info("Saving data...", Log.LVL_INFO);
	local save_table = {};

	/* If the script isn't yet initialized, we can't retrieve data
	 * from GoalTown instances. Thus, simply use the original
	 * loaded table. Otherwise we build the table with town data.
	 */


	if (!this._gs_init_done) {
		save_table.town_data_table <- ::_town_data_table;
	} else {
		foreach (i, town in this._towns)
		{
			::_town_data_table[town.id] <- town.SavingTownData();
		}
		save_table.town_data_table <- ::_town_data_table;
	}

	// Also store a savegame version flag
	save_table.save_version <- ::_current_save_version;

	return save_table;
}

function MainClass::Load(version, saved_data)
{
	Log.Info("Loading data...", Log.LVL_INFO);
	// Loading town data. Only load data if the savegame version matches.
	if (saved_data.rawin("save_version")) {
		::_loaded_save_version = saved_data.save_version;
		if (saved_data.save_version == ::_current_save_version) {
			foreach (townid, town_data in saved_data.town_data_table) {
				::_town_data_table[townid] <- town_data;
			}
		}
		else {
			Log.Info("Data format doesn't match with current version. Resetting.", Log.LVL_INFO);
		}
	}
}

/* Make a squirrel array of GoalTown instances (towns_array). For each
 * town, an instance of GoalTown is created to store the data related
 * to that town.
 */
function MainClass::CreateTownList()
{
	local towns_list = GSTownList();
	local towns_array = [];

	foreach (t, _ in towns_list) {
		towns_array.append(GoalTown(t));
	}

	return towns_array;
}

/* Function called on town creation. We need to add a now GoalTown
 * instance to the this.towns array. The array order doesn't matter,
 * since we never use the array index, only its values.
 */
function MainClass::UpdateTownList(town_id)
{
	this._towns.append(GoalTown(town_id));
	Log.Info("New town founded: "+GSTown.GetName(town_id)+" (id: "+town_id+")", Log.LVL_DEBUG);
}

/* Function called periodically (each 60 ticks) to manage
 * towns and other stuff.
 */
function MainClass::ManageTowns()
{
	// Run the daily functions
	local date = GSDate.GetCurrentDate();
	local diff_date = date - this._current_date;
	if (diff_date == 0) {
		return;
	} else {
		this._current_date = date;
	}

	// Run the monthly functions
	local month = GSDate.GetMonth(date);
	local diff_month = month - this._current_month;
	if (diff_month == 0) {
		return;
	} else {
		local month_tick = GSController.GetTick();
		Log.Info("Starting Monthly Updates...", Log.LVL_INFO);
		
		foreach (town in this._towns) {
			town.MonthlyManageTown();
		}
		
		this._current_month = month;
		local month_tick_duration = GSController.GetTick() - month_tick;
		Log.Info("Monthly Update took "+month_tick_duration+" ticks.", Log.LVL_DEBUG);
	}

	// Run the yearly functions - Nothing to do for now, so we leave it out
	/*local year = GSDate.GetYear(date);
	local diff_year = year - this._current_year;
	if ( diff_year == 0) return;
	else
	{
		GSLog.Info("Starting Yearly Updates");
		this._current_year = year
	}*/
}
//TLIB
/*
 * Called by our main loop when a new month has been reached.
 */
function MainClass::EndOfMonth(year)
{
	this.industryBuilder.onNewMonth(year);
}

/*
 * Called by our main loop when a new year has been reached.
 */
function MainClass::EndOfYear(year)
{
	this.industryBuilder.onNewYear(year);
}
