/*
 * This file is part of Renewed City Growth, a GameScript for OpenTTD.
 * Copyright (C) 2013-2014 keoz
 *
 * It's free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the
 * Free Software Foundation, version 2 of the License.
 *
 */


/*
 * For each town in the game, an instance of this class is created.
 * It holds the data related to a specific town.
 */
class GoalTown
{
	id = null;                  // Town id
	sign_id = null;             // Id for extra text under town name
	monitor_check = null;       // Whether the town is already under monitoring. True if town exchanges pax.
	last_pax_delivery = null;   // Date of last pax supply
	town_supplied = null;       // Last monthly supply per cargo type
	town_goals_cat = null;      // Town goals per cargo category
	town_supplied_cat = null;   // Last monthly supply per cargo category (for categories: see InitCargoLists())
	town_stockpiled_cat = null; // Stockpiled cargos per cargo category
	tgr_array = null;           // Town growth rate array
	tgr_array_len = null;       // Town growth rate array lenght
	tgr_average = null;         // Town growth rate average, calculated from the array

	constructor(town_id) {
		this.id = town_id;
		this.tgr_array_len = 8;
		this.tgr_average = null;
		/* If there isn't saved data for the towns, we
		 * initialize them. Otherwise, we load saved data.
		 */
		if (!::_load_saved_data || !::_town_data_table.rawin(this.id)) {
			this.sign_id = -1;
			this.monitor_check = false;
			this.last_pax_delivery = null;
			this.town_supplied = array(::CargoTypeNum, 0);
			this.town_goals_cat = array(::CargoCatNum, 0);
			this.town_supplied_cat = array(::CargoCatNum, 0);
			this.town_stockpiled_cat = array(::CargoCatNum, 0);
			this.tgr_array = array(tgr_array_len, 0);
			GSTown.SetGrowthRate(this.id, GSTown.TOWN_GROWTH_NONE);
			GSTown.SetText(this.id, GSText(GSText.STR_TOWNBOX_NOGROWTH));
		} else {
			this.sign_id = ::_town_data_table[this.id].sign_id;
			// Though saved as bool, monitor_check is reloaded as int (why ?). We convert it back.
			this.monitor_check = (::_town_data_table[this.id].monitor_check != 0);
			this.last_pax_delivery = ::_town_data_table[this.id].last_pax_delivery;
			this.town_supplied = ::_town_data_table[this.id].town_supplied;
			this.town_goals_cat = ::_town_data_table[this.id].town_goals_cat;
			this.town_supplied_cat = ::_town_data_table[this.id].town_supplied_cat;
			this.town_stockpiled_cat = ::_town_data_table[this.id].town_stockpiled_cat;
			this.tgr_array = ::_town_data_table[this.id].tgr_array;
		}
	}
}

function GoalTown::SavingTownData()
{
	/* IMPORTANT: if anything of the saved data changes here, we
	 * need to update the ::_save_version flag in MainClass'
	 * constructor.
	 */
	local town_data = {};
	town_data.sign_id <- this.sign_id;
	town_data.monitor_check <- this.monitor_check;
	town_data.last_pax_delivery <- this.last_pax_delivery;
	town_data.town_supplied <- this.town_supplied;
	town_data.town_goals_cat <- this.town_goals_cat;
	town_data.town_supplied_cat <- this.town_supplied_cat;
	town_data.town_stockpiled_cat <- this.town_stockpiled_cat;
	town_data.tgr_array <- this.tgr_array;
	return town_data;
}

/* Main town management function. Called each month. */
function GoalTown::MonthlyManageTown()
{
	local sumgoals = 0;
	local goaldiff = 0;
	local goaldiffpercent = 0.0;
	local curpop = GSTown.GetPopulation(this.id);
	local parsed_cat = 0;	// index of parsed category
	local new_town_growth_rate = null;
	// Defining difficulty factor
	local d_percent = GSController.GetSetting("goal_scale_factor");
	local d_factor = d_percent > 0? d_percent / 100.0 : 1.0; // Use 1.0 if setting is missing
	// Clearing the arrays
	this.town_supplied = array(::CargoTypeNum, 0);
	this.town_supplied_cat = array(::CargoCatNum, 0);
	this.town_goals_cat = array(::CargoCatNum, 0);

	// Checking whether we should enable monitoring and perform calculations
	if (!this.monitor_check && !this.CheckMonitoring(false)) return;

	// Checking cargo delivery
	for (local i = 0; i < ::CargoTypeNum; i++) {
		for (local cid = GSCompany.COMPANY_FIRST; cid <= GSCompany.COMPANY_LAST; cid++) {
			if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID) {
				town_supplied[i] += GSCargoMonitor.GetTownDeliveryAmount
					(cid, ::CargoList[i], this.id, true);
				if (town_supplied[i] > 0) this.DebugCargoSupplied(i); // Debug info: cargo supplied
			}
		}

		// On first iteration, if there isn't pax supply, check whether we stop monitoring
		if (i == 0 && town_supplied[i] == 0 && !this.CheckMonitoring(true)) return;

		// Summing up cargo supply per each cargo categories.
		if (this.town_supplied[i] > 0) {
			local is_found = false;
			while (parsed_cat < ::CargoCatNum) {  // On parse les groupes un par un.
				for (local k = 0; k < ::CargoListCat[parsed_cat].len(); k++) {
					if (::CargoList[i] == ::CargoListCat[parsed_cat][k]) {
						this.town_supplied_cat[parsed_cat] += this.town_supplied[i];
						is_found = true;
						if ((k+1) == ::CargoListCat[parsed_cat].len()) {
							parsed_cat++;
						}
						
						break;
					}
				}

				if (is_found == true) {
					break;
				}

				parsed_cat++;
			}
		}
	}

	// Calculating goals
	for (local i = 0; i < CargoCatNum && curpop > ::CargoMinPopDemand[i]; i++) {
		this.town_goals_cat[i] = max((((curpop  - ::CargoMinPopDemand[i]).tofloat() / 1000)
					    * ::CargoPermille[i]
					    * d_factor).tointeger(),1);
	}

	// If town's population is too low to calculate a goal, it is set to 1
	if (this.town_goals_cat[0] < 1) this.town_goals_cat[0] = 1;

	// Calculating global goal and achievement
	for (local i = 0; i < CargoCatNum; i++) {
		// Supplied stuff are summed with stockpiled stuff. sumgoals is incremented for each category.
		if (this.town_goals_cat[i] > 0) {
			sumgoals += this.town_goals_cat[i];
			this.town_stockpiled_cat[i] += this.town_supplied_cat[i];
		}

		// Stockpiles are updated. goaldiff measures how many stuff is missing for goal achievement.
		if (this.town_stockpiled_cat[i] < this.town_goals_cat[i]) {
			goaldiff += (this.town_goals_cat[i] - this.town_stockpiled_cat[i]);
			this.town_stockpiled_cat[i] = 0;
		} else {
			// If stockpiled is bigger than required, we cut off the required part
			this.town_stockpiled_cat[i] = ((this.town_stockpiled_cat[i]- this.town_goals_cat[i])
						     * (1 - ::CargoDecay[i])).tointeger();
			// Don't stockpile more than: (cargo category) * 10;
			if (this.town_stockpiled_cat[i] > 300 &&
			    this.town_stockpiled_cat[i] > this.town_goals_cat[i] * 10) {
				this.town_stockpiled_cat[i] = this.town_goals_cat[i] * 10;
			}
		}
		this.DebugCargoCatInfo(i) // Debug info: print stockpiled/supplied/goal per category
	}

	/* Here we calculate the number of days until the next
	 * growth. The important stuff happens here. Other possible
	 * formulas for calculating the new town growth rate are:
	 * new_town_growth_rate = (max_town_growth_rate * (1+(10/(1-goaldiffpercent)-10))).tointeger();
	 * new_town_growth_rate = (max_town_growth_rate/(1-goaldiffpercent*2)).tointeger();
	 */
	goaldiffpercent = (goaldiff.tofloat() / sumgoals.tofloat()); // Convert the difference to a percent
	this.DebugGoalsResult(sumgoals, goaldiff, goaldiffpercent);  // Debug info about general goal results
	if (goaldiffpercent < 0.5) {
		local max_town_growth_rate = 50000 / (100 + curpop.tofloat());
		new_town_growth_rate = (max_town_growth_rate * (1+(3/(1-goaldiffpercent*2)-3))).tointeger();
	} else if (goaldiffpercent >= 0.5 || new_town_growth_rate > 10000) {
		new_town_growth_rate = 10000;
	}

	// Defining the new town growth rate, calculated as the moving average of the TGR array
	local sum_array = 0;
	local i = 0;
	while (this.tgr_array[i] > 0) {
		sum_array += this.tgr_array[i];
		i++;
	}
	this.tgr_array[i] = new_town_growth_rate;
	sum_array += this.tgr_array[i];
	this.tgr_average = (sum_array/(i+1)).tointeger();
	GSTown.SetGrowthRate(this.id, this.tgr_average);

	// Shift the array by one element when full
	this.DebugTgrArray() // Debug info: print the array's content
	if (this.tgr_array[this.tgr_array_len-1] > 0) {
		this.tgr_array = this.tgr_array.slice(1); // efface element [0] de l'array
		this.tgr_array.push(0);                   // ajoute 0 en dernier element
	}

	this.UpdateTownText();
}

/* Returns true if the city needs to be monitored, false otherwise. */
function GoalTown::CheckMonitoring(monitored)
{
	local paxcheck = 0;
	for (local cid = GSCompany.COMPANY_FIRST; cid <= GSCompany.COMPANY_LAST; cid++) {
		if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID) {
			paxcheck += GSCargoMonitor.GetTownPickupAmount(cid, ::CargoList[0], this.id, true);
			if (paxcheck > 0) break;
		}
	}

	if (!monitored) {
		/* For unmonitored cities: checking whether they
		 * supply pax. If they do, monitoring is enabled.
		 */
		if (paxcheck == 0) {
			return false;
		} else {
			this.monitor_check = true;
			this.last_pax_delivery = GSDate.GetCurrentDate();
			Log.Info("City of "+GSTown.GetName(this.id)+" now monitored", Log.LVL_DEBUG);
			return true;
		}
	} else {
		/* For monitored cities: if there isn't passengers
		 * delivery to the city anymore (in which case only
		 * this function is called from MonthlyManageTown())
		 * AND the town doesn't supply passengers since more
		 * than 6 months (which is checked here), we stop the
		 * monitoring.
		 */
		if (paxcheck > 0) {
			this.last_pax_delivery = GSDate.GetCurrentDate();
			return true;
		}
//assertion
		if (this.last_pax_delivery == null) {
			return false;
		} 

		if ((GSDate.GetCurrentDate()-this.last_pax_delivery) < 365) {
			return true;
		} else {
			GSTown.SetGrowthRate(this.id, GSTown.TOWN_GROWTH_NONE);
			this.monitor_check = false;
			this.town_supplied = array(::CargoTypeNum, 0);
			this.town_stockpiled_cat = array(::CargoCatNum, 0);
			this.tgr_array = array(tgr_array_len, 0);
			this.tgr_average = null;
			this.StopMonitors();
			this.RemoveTownText();
			Log.Info("City of "+GSTown.GetName(this.id)+" is not monitored anymore", Log.LVL_DEBUG);
			return false;
		}
	}
}

function GoalTown::StopMonitors()
{
	for (local i = 0; i< ::CargoTypeNum; i++) {
		for (local cid = GSCompany.COMPANY_FIRST; cid <= GSCompany.COMPANY_LAST; cid++) {
			if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID) {
				GSCargoMonitor.GetTownDeliveryAmount(cid, ::CargoList[i],this.id,false);
			}
		}
	}
}

function GoalTown::UpdateTownText()
{
	// Add a sign by the town to display the current growth
	local sign_text = TownSignText();
	if (GSSign.IsValidSign(this.sign_id)) {
		GSSign.SetName(this.sign_id, sign_text);
	} else {
		this.sign_id = GSSign.BuildSign(GSTown.GetLocation(this.id), sign_text);
	}

	// Set the towns text box
	GSTown.SetText(this.id, this.TownBoxText(true));
}

function GoalTown::RemoveTownText()
{
	// Cleaning signs on the map
	if (GSSign.IsValidSign(this.sign_id)) {
		GSSign.RemoveSign(this.sign_id);
		this.sign_id = -1;
	}

	// Cleaning town's info box
	GSTown.SetText(this.id, this.TownBoxText(false));
}
